/*
* KAMATSU'S CLASS LIBRARY
* Version 0.45
*  
* Created by Kamatsu
* 
* Released under the MIT license.
* 
* Copyright (C) 2007 Liam O'Connor-Davis e.a. Only a couple of rights reserved.
* Feel free to use/modify this for your projects, provided this copyright notice
* remains intact.
*
* Also, no warranty. Either expressed or implied, including the implied warranties
* of merchantability or fitness for a particular purpose. 
*
* Enjoy.
*
*/

/*

kclScreenElementClasses.js: Screen Elements. Good for Non-RPGs that don't use the map engine.

*/

try {
    LibraryRegistry.abortIfNotInstalled("Spheritype",1.1,false);
}
catch (ex) {
    Abort("Spheritype Not Installed!");
}


//Prototype Extension to Math probably not worth depending on the whole lot of them.
Math.pi = 3.1415926535897931;

LibraryRegistry.abortIfNotInstalled("Kamatsu's Class Library: Declares",0.45, true); //Versions must equate :), hence the strict arg.
LibraryRegistry.abortIfNotInstalled("Kamatsu's Class Library: Wrapper Classes",0.45, true); //Versions must equate :), hence the strict arg.
LibraryRegistry.addLibraryData("Kamatsu's Class Library: Screen Element Classes",0.45);





/* OBJECT: ScreenElement
    Parameters: 
       * x, y - Coordinates to blit image or surface
       * background - background to place behind image.
       * image - image to render
       
       
    Non-parametised Properties:
       * opacity - Opacity with which to render.
       * visible - Whether to draw image/surface or not.
       * flipScreen - flipscreen on redraw..
       * rotateImage - Whether or not to rotate image by angle (useful for polar coordinates)
       * preRender, postRender - first-class render functions
       
    Inherits from: Nothing

    An object for moveable "screen elements". Basically an image with an x,y, and angle property for rendering. Can use polar coordinates and movement using the go() method.
*/
var ScreenElement = Class.extend({

    constructor: function (image, x, y, angle, background) {
    
         if (x) this.x = x; else this.x = 0;
         if (y) this.y = y; else this.y = 0;

         if (background) this.background = background; else this.background = BACKGROUND_NONE;
         if (image) {
             this.image = image;
         } else Abort("Must provide an image or surface to ScreenElement");
    },
    
    //Reminders. Props defined in constructor.
    opacity: 255,
    x: undefined,
    y: undefined,
    angle: 0,
    visible: true,
    image: undefined,
    flipScreen: false,
    rotateImage: true,
    
    preRender: undefined,
    postRender: undefined,
      
    /* METHOD: getWidth
    
       Returns the width of the image
    */
    getWidth: function() {
        return this.image.getWidth();
    
    },
    
    /* METHOD: getHeight
    
       Returns the Height of the image
    */
    getWidth: function() {
        return this.image.getHeight();
    
    },
 
    /* METHOD: go
    
       Parameters: Amount - amount of pixels to move
       
       Move forward or backward along the direction produced by the rotation. 
    */   
    go: function(amount) {
        if (!amount) amount = 1;
        var vx = amount * Math.cos(this.angle);
        var vy = amount * Math.sin(this.angle);
        this.x += vx;
        this.y += vy;
    },
    
    /* METHOD: drawBackground
    
       Parameters: None
       
       Mostly-internal function for drawing backgrounds. 
    */   
    drawBackground: function () {
        if (this.background == BACKGROUND_RENDERMAP) {
            RenderMap();
        }
        else if (this.background != BACKGROUND_NONE) {
            this.background.blit(0,0);
        }
    },
    
    /* METHOD: update
    
       Parameters: none
       
       Redo drawing, flipping if necessary. 
    */   
    update: function () {
        this.drawBackground();
        if (this.preRender) this.preRender();
        //Rectify angles to be within one revolution
        while (!(this.angle >= 0 && this.angle <= 2 * Math.pi)) {
            if (this.angle < 0) {
                this.angle = 2 * Math.pi + this.angle;
            }
            
            this.angle %= 2 * Math.pi;
         }
        
        if (this.image.toImage) { //i.e we have a surface
            //try { this.image.setAlpha(this.opacity);} catch (ex) {}
        }
        else if (this.image.toSurface) { //i.e we have an image
        
             this.image.mask = new Color(this.image.mask.getRed(),this.image.mask.getGreen(),this.image.mask.getBlue(),this.opacity);
        
        }
        
        if (this.visible) {
            if (this.rotateImage == true) {
                angle = this.angle;
                 
            }
            else {
                angle = 0;
            }
            if (this.image.toImage) {
                var temp = this.image.clone();
                temp.rotate(angle,true); 
                temp.blit(this.x,this.y);
            }
            else this.image.rotateBlit(this.x,this.y,angle);
         
        
        }
        if (this.postRender) this.postRender();
        if (this.flipScreen) FlipScreen();
    
    }
});

/* OBJECT: TextElement
    Parameters: 
       * x, y - Coordinates to blit image or surface
       * text - Text to draw
       * background - background to place behind image.
       * font - font to use for text.
       
    Non-parametised Properties:
       * opacity - Opacity with which to render.
       * visible - Whether to draw image/surface or not.
       * flipScreen - flip the screen on drawing..
       * angle - for polar movement
       * preRender, postRender - first-class render functions
       
    Inherits from: Nothing

    An object for moveable "screen elements". Basically an image with an x,y, and angle property for rendering. Can use polar coordinates and movement using the go() method.
*/
var TextElement = Class.extend({

    constructor: function (text,font, x, y, background) {
    
         if (x) this.x = x; else this.x = 0;
         if (y) this.y = y; else this.y = 0;

         if (background) this.background = background; else this.background = BACKGROUND_NONE;
         if (text) this.text = text; else this.text = "";
         if (font) this.font = font; else this.font = new Font();
         
    },
    
    //Reminders. Props defined in constructor.
    opacity: 255,
    x: undefined,
    y: undefined,
    font: undefined,
    text: undefined,
    visible: true,
    flipScreen: false,
    angle: 0,
    preRender: undefined,
    postRender: undefined,
    
    /* METHOD: go
    
       Parameters: Amount - amount of pixels to move
       
       Move forward or backward along the direction produced by the rotation. 
    */   
    go: function(amount) {
        if (!amount) amount = 1;
        var vx = amount * Math.cos(this.angle);
        var vy = amount * Math.sin(this.angle);
        this.x += vx;
        this.y += vy;
    },
    
    
    
    /* METHOD: getWidth
    
       Returns the width of the text
    */
    getWidth: function() {
        return this.font.getStringWidth(this.text);
    
    },
    
    /* METHOD: getHeight
    
       Returns the Height of the text
    */
    getWidth: function() {
        return this.font.getHeight();
    
    },
    
    /* METHOD: drawBackground
    
       Parameters: None
       
       Mostly-internal function for drawing backgrounds. 
    */       
    drawBackground: function () {
        if (this.background == BACKGROUND_RENDERMAP) {
            RenderMap();
        }
        else if (this.background != BACKGROUND_NONE) {
            this.background.blit(0,0);
        }
    },
    
    
    /* METHOD: update
    
       Parameters: none
       
       Redo drawing, flipping if necessary. 
    */ 
    update: function () {
        this.drawBackground();

        if (this.preRender) this.preRender()    
        if (this.font.getColorMask().getAlpha() != this.opacity) {
            this.font.setColorMask(new Color(this.font.getColorMask().getRed(), this.font.getColorMask().getGreen(), this.font.getColorMask().getBlue(), this.opacity));
        }
        
        if (this.visible) {
        
            this.font.drawText(this.x,this.y,this.text);
        }
        if (this.postRender) this.postRender()    
        if (this.flipScreen) FlipScreen();
    
    }
    
 });
 
